<?php
namespace app\models;

use Yii;
use yii\base\NotSupportedException;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;
/**
 * User model
 *
 * @property integer $id
 * @property string $username
 * @property string $password_hash
 * @property string $password_reset_token
 * @property string $email
 * @property string $auth_key
 * @property integer $status
 * @property integer $created_at
 * @property integer $updated_at
 * @property string $password write-only password
 */
class User extends ActiveRecord implements IdentityInterface
{
    const Status_Deleted = 0;
    const Status_Active = 10;

    const ROLE_MANAGER = 1;
    const ROLE_EXECUTIVE = 2;
    const ROLE_DEPARTMENTMANAGER = 3;
    const ROLE_HUMANRESOURCES = 4;

    public static $roleOptions = [
        self::ROLE_MANAGER =>"Gerente",
        self::ROLE_EXECUTIVE =>"Ejecutivo",
        self::ROLE_DEPARTMENTMANAGER =>"Cliente/Jefe de Departamento",
        self::ROLE_HUMANRESOURCES =>"Cliente/Recursos Humanos",
    ];
    public static $statusOptions = [
        self::Status_Active =>"Activo",
        self::Status_Deleted =>"Inactivo",
    ];
    public $comparePassword;
    public $auxCurrentCompany;
    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'Users';
    }
    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            TimestampBehavior::className(),
        ];
    }
    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['username','firstName','lastName','email','status','role'],'required'],
            ['email','email','message'=>'El <strong>correo electronico</strong> no es valido'],
            ['password_hash','compare','operator'=>'===','compareAttribute'=>'comparePassword','message'=>'Las <strong>contraseñas</strong> no coinciden!','on'=>['create','changePassword']],
            ['username', 'unique', 'message' => "Nombre de usuario ya en uso"],
            ['status', 'default', 'value' => self::Status_Active],
            ['status', 'in', 'range' => [self::Status_Active, self::Status_Deleted]],
            ['comparePassword','safe'],
        ];
    }
    /**
     * @inheritdoc
     */
    public static function findIdentity($id)
    {
        return static::findOne(['id' => $id, 'status' => self::Status_Active]);
    }
    /**
     * @inheritdoc
     */
    public static function findIdentityByAccessToken($token, $type = null)
    {
        throw new NotSupportedException('"findIdentityByAccessToken" is not implemented.');
    }
    /**
     * Finds user by username
     *
     * @param string $username
     * @return static|null
     */
    public static function findByUsername($username)
    {
        return static::findOne(['username' => $username, 'status' => self::Status_Active]);
    }
    /**
     * Finds user by password reset token
     *
     * @param string $token password reset token
     * @return static|null
     */
    public static function findByPasswordResetToken($token)
    {
        if (!static::isPasswordResetTokenValid($token)) {
            return null;
        }

        return static::findOne([
            'password_reset_token' => $token,
            'status' => self::Status_Active,
        ]);
    }
    /**
     * Finds out if password reset token is valid
     *
     * @param string $token password reset token
     * @return boolean
     */
    public static function isPasswordResetTokenValid($token)
    {
        if (empty($token)) {
            return false;
        }
        $expire = Yii::$app->params['user.passwordResetTokenExpire'];
        $parts = explode('_', $token);
        $timestamp = (int) end($parts);
        return $timestamp + $expire >= time();
    }
    /**
     * @inheritdoc
     */
    public function getId()
    {
        return $this->getPrimaryKey();
    }
    /**
     * @inheritdoc
     */
    public function getAuthKey()
    {
        return $this->auth_key;
    }
    /**
     * @inheritdoc
     */
    public function validateAuthKey($authKey)
    {
        return $this->getAuthKey() === $authKey;
    }
    /**
     * Validates password
     *
     * @param string $password password to validate
     * @return boolean if password provided is valid for current user
     */
    public function validatePassword($password)
    {
        // return Yii::$app->security->validatePassword($password,$this->password_hash);
        return md5($password)===($this->password_hash);
    }

    /**
     * Generates password hash from password and sets it to the model
     *
     * @param string $password
     */
    public function setPassword($password)
    {
        // $this->password_hash = Yii::$app->security->generatePasswordHash($password);
        $this->password_hash = md5($password);
    }

    /**
     * Generates "remember me" authentication key
     */
    public function generateAuthKey()
    {
        $this->auth_key = Yii::$app->security->generateRandomString();
    }

    /**
     * Generates new password reset token
     */
    public function generatePasswordResetToken()
    {
        $this->password_reset_token = Yii::$app->security->generateRandomString() . '_' . time();
    }

    /**
     * Removes password reset token
     */
    public function removePasswordResetToken()
    {
        $this->password_reset_token = null;
    }
    public function getStatusArray()
    {
        return [
            self::Status_Active =>'Activo',
            self::Status_Deleted =>'Inactivo',
        ];
    }
    public function beforeSave($insert)
    {
        if(parent::beforeSave($insert)){
            if(in_array($this->scenario,['create','changePassword'])){
                $this->setPassword($this->password_hash);
            }
            return true;
        }else{
            return false;
        }
    }
    public function hasCompany($id)
    {
        return $this->getCompanies()->andWhere(['id'=>$id])->count()>0;
    }
    public function hasDepartment($departmentId,$companyId)
    {
        return $this->getPermitDepartments()->andWhere(['idDepartment'=>$departmentId,'idCompany'=>$companyId])->count()>0;
    }
    public function getCompanies()
    {
        return $this->hasMany(Company::classname(),['id'=>'idCompany'])->viaTable('UsersCompanies',['idUser'=>'id'])->orderBy('name ASC');
    }
    public function getDepartments($idCompany)
    {
        return $this->
        hasMany(Department::classname(),['id'=>'idDepartment'])
        ->viaTable('PermitsDepartments',['idUser'=>'id'],function($query) use ($idCompany){
            $query->onCondition(['idCompany'=>$idCompany]);
        });
        // $deps = [];
        // $permits = $this->getPermitDepartments()->joinWith('department')->andWhere(['idCompany'=>$idCompany])->orderBy('Departments.name ASC')->all();
        // foreach ($permits as $pd) $deps[]=$pd->department;
        // return $deps;
    }
    public function getPermitDepartments()
    {
        return $this->hasMany(PermitDepartment::classname(),['idUser'=>'id']);
    }
    public function getFullName()
    {
        return $this->firstName." ".$this->lastName;
    }
    public function isAdmin()
    {
        return in_array($this->role,[self::ROLE_MANAGER,self::ROLE_EXECUTIVE]);
    }
    public function getCurrentCompany()
    {
        if(!isset($this->auxCurrentCompany)){
            $this->auxCurrentCompany = Company::findOne(Yii::$app->session->get('currentCompanyId'));
        }
        return $this->auxCurrentCompany;        
    }
}
