<?php 
namespace app\models;
use Yii;
use app\models\BaseModel;
use app\components\EmployeesComponent;
use app\components\SettingsComponent;

class Employee extends BaseModel{
    const STATUS_ACTIVE = 1;
    const STATUS_INACTIVE = 2;
    const STATUS_READMISSION = 3;

    const MARITALSTATUS_MARRIED = 1;
    const MARITALSTATUS_SINGLE = 2;
    const MARITALSTATUS_COMMONLAWUNION = 3;
    const MARITALSTATUS_DIVORCED = 4;
    const MARITALSTATUS_WIDOWER = 5;

    const Gender_Male = 'M';
    const Gender_Female = 'F';

    public $compareEmail;
    public function init()
    {
        if($this->isNewRecord){
            $this->refId = EmployeesComponent::getNextId();
            $this->creationUser=Yii::$app->user->identity->id;
            $this->status = self::STATUS_ACTIVE;
            $this->minimumSalary = SettingsComponent::getMinimumSalary();
            $this->localEmployee = 1;
        }
        return parent::init();
    }
    public static function tableName()
    {
        return 'Employees';
    }
    public function rules()
    {
        return [
            [['firstName','email','birthDate','address','addressOfBirth','gender','city','state','maritalStatus','idCompany','idDepartment','idJob','entryDateCompany','entryDateSaav','salaries'],'required','message'=>'Campo <strong>{attribute}</strong> obligatorio'],
            ['idCompany','exist','targetClass'=>"\app\models\Company",'targetAttribute'=>'id','message'=>'El campo <strong>{attribute}</strong> no es valido'],
            ['idDepartment','exist','targetClass'=>"\app\models\Department",'targetAttribute'=>'id','message'=>'El campo <strong>{attribute}</strong> no es valido'],
            ['idJob','exist','targetClass'=>"\app\models\Job",'targetAttribute'=>'id','message'=>'El campo <strong>{attribute}</strong> no es valido'],
            ['refId','unique','message'=>"El id {value} ya ha sido utilizado"],
            [['birthDate','entryDateCompany','entryDateSaav'],'date','format'=>'yyyy-mm-dd','message'=>"La fecha del campo <strong>{attribute}</strong> no es valida"],
            ['gender','in','range'=>[self::Gender_Female,self::Gender_Male],'message'=>'El campo <strong>{attribute}</strong> no es valido'],
            ['maritalStatus','in','range'=>[self::MARITALSTATUS_SINGLE,self::MARITALSTATUS_MARRIED,self::MARITALSTATUS_COMMONLAWUNION,self::MARITALSTATUS_DIVORCED,self::MARITALSTATUS_WIDOWER],'message'=>'El campo <strong>{attribute}</strong> no es valido'],
            [['curp','rfc','nss','postalCode'],'required','when'=>function($model){return $model->localEmployee;},'message'=>'Campo <strong>{attribute}</strong> obligatorio'],
            ['curp','string','length'=>18,'notEqual'=>'El campo CURP debe ser de 18 caracteres'],
            ['rfc','string','length'=>13,'notEqual'=>'El campo RFC debe ser de 13 caracteres'],
            ['nss','string','length'=>11,'notEqual'=>'El campo NSS debe ser de 11 caracteres'],
            ['email','email','message'=>'Ingresa un Correo Electrónico valido'],
            ['email','compare','operator'=>'===','compareAttribute'=>'compareEmail','message'=>'Los correos electrónicos no coinciden!','on'=>'create'],
            [['lastName1','lastName2','refId','localEmployee','phone','mobilePhone','compareEmail','bankAccountBank','bankAccountAccountNumber','bankAccountCardNumber','bankAccountClabe','status','breakDays','coupleName'],'safe'],
        ];
    }
    public function getFullName()
    {
        return $this->firstName." ".$this->lastName1." ".$this->lastName2;
    }
    public function getFullNameInverse()
    {
        return $this->lastName1." ".$this->lastName2." ".$this->firstName;
    }
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if($this->isNewRecord) $this->refId = EmployeesComponent::getNextId();
            $this->updateDate = date('Y-m-d H:i:s');
            $this->updateUser = Yii::$app->user->identity->id;
            return true;
        } else {
            return false;
        }
    }
    public function attributeLabels()
    {
        return [
            'firstName'=>"Nombres",
            'lastName1'=>"Apellido paterno",
            'lastName2'=>"Apellido materno",
            'email'=>"Correo electrónico",
            'birthDate'=>"Fecha de Nacimiento",
            'curp'=>"CURP",
            'rfc'=>"RFC",
            'nss'=>"Seguro social",
            'address'=>"Dirección",
            'addressOfBirth'=>"Lugar de nacimiento",
            'city'=>"Ciudad",
            'state'=>"Estado",
            'gender'=>"Sexo",
            'postalCode'=>"Código postal",
            'refId'=>"ID",
            'idCompany'=>"Empresa",
            'idJob'=>"Puesto",
            'idDepartment'=>"Departamento",
            'entryDateCompany'=>"Fecha ingreso a empresa",
            'maritalStatus'=>"Estado civil",
        ];
    }
    public static function getGenderOptions()
    {
        return [
            self::Gender_Male => 'Masculino',
            self::Gender_Female => 'Femenino',
        ];
    }
    public static function getMaritalStatusOptions()
    {
        return [
            self::MARITALSTATUS_SINGLE => "Soltero(a)",
            self::MARITALSTATUS_MARRIED => "Casado(a)",
            self::MARITALSTATUS_COMMONLAWUNION => "Unión libre",
            self::MARITALSTATUS_DIVORCED => 'Divorciado',
            self::MARITALSTATUS_WIDOWER => 'Viudo',
        ];
    }
    public static function getStatusOptions()
    {
        return [
            self::STATUS_ACTIVE => "ALTA",
            self::STATUS_INACTIVE => "BAJA",
            self::STATUS_READMISSION => "REINGRESO",
        ];
    }
    public function getJsonDocs()
    {
        $aux = json_decode($this->jsonDocs);
        $aux = $aux===null?[]:$aux;
        return $aux;
    }
    public function getDocs($localEmployee=true)
    {
        if($localEmployee){
            return [
                "Acta de nacimiento",
                "Identificación oficial",
                "Comprobante de domicilio",
                "Cédula profesional",
                "CURP",
                "RFC",
                "Correo electrónico",
                "Número de IMSS",
            ];            
        }
        return [
            "Acta de nacimiento",
            "Certificado",
            "Comprobante de estudios apostillados",
            "Documento migratorio vigente",
            "Correo electrónico",
        ];            
    }
    public function getContracts()
    {
        return $this->hasMany(Contract::classname(),['idEmployee'=>'id']);
    }
    public function getLastContract()
    {
        return $this->getContracts()->orderBy('id DESC')->one();     
    }
    public function getLastChange()
    {
        return $this->getEmployeeChanges()->orderBy('id DESC')->one();     
    }
    public function getFirstChange()
    {
        // debe ser la fecha de alta
        return $this->getEmployeeChanges()->orderBy('id ASC')->one();   
    }
    public function getMaritalStatusName()
    {
        return self::getMaritalStatusOptions()[$this->maritalStatus];
    }
    public function getGenderName()
    {
        return self::getGenderOptions()[$this->gender];
    }
    public function getCompany()
    {
        return $this->hasOne(Company::classname(),['id'=>'idCompany']);
    }
    public function getIncidents()
    {
        return $this->hasMany(Incident::classname(),['idEmployee'=>'id']);
    }
    public function getDepartment()
    {
        return $this->hasOne(Department::classname(),['id'=>'idDepartment']);
    }
    public function getJob()
    {
        return $this->hasOne(Job::classname(),['id'=>'idJob']);
    }
    public function getPayrollConcepts()
    {
        return $this->hasMany(PayrollConcept::classname(),['idEmployee'=>'id']);
    }
    public function getEmployeeChanges()
    {
        return $this->hasMany(EmployeeChange::classname(),['idEmployee'=>'id']);
    }
    public function afterSave($insert, $changedAttributes)
    {
        // EmployeesComponent::saveState($this);
        parent::afterSave($insert, $changedAttributes);
    }
    public function getVacationsRequests()
    {
        return $this->hasMany(VacationRequest::classname(),['idEmployee'=>'id']);
    }
    public function getFiscalSalary()
    {
        return $this->salaries*$this->minimumSalary;
    }
    public function getFiscalMonthlySalary()
    {
        return $this->fiscalSalary*365/12;
    }
    public function getInactiveDate($startDate=false,$endDate=false)
    {
        // busca ultimo movimiento de baja
        $q = $this->getEmployeeChanges()
        ->where(['changeType'=>EmployeeChange::CHANGE_TYPE_STATUS_INACTIVE])
        ->orderBy('id DESC');
        if($startDate!==false && $endDate!==false) $q->andWhere("applyDate >= '$startDate' AND applyDate <= '$endDate'");
        $change = $q->one();
        return $change?$change->applyDate:'0000-00-00';
    }
    public function getReadmissionDate($startDate=false,$endDate=false)
    {
        // busca ultimo movimiento de baja
        $q = $this->getEmployeeChanges()
        ->where(['changeType'=>EmployeeChange::CHANGE_TYPE_STATUS_READMISSION])
        ->orderBy('id DESC');
        if($startDate!==false && $endDate!==false) $q->andWhere("applyDate >= '$startDate' AND applyDate <= '$endDate'");
        $change = $q->one();
        return $change?$change->applyDate:'0000-00-00';
    }
    public function getStatusToUp()
    {
        return $this->status==self::STATUS_INACTIVE;
    }
    public function getBreakDays()
    {
        return $this->breakDays!=""?str_split($this->breakDays):[];
    }
    public function getChildrens()
    {
        $childrens = json_decode($this->jsonChildrens);
        return $childrens?$childrens:[];
    }
}
?>